'=================================================================================================================================
'    Script Name:    SDS_Addresses_Find_And_Export.vbs
'    Version:        0.3
'    Release Date:   08/22/2025
'    Purpose:        Find and export SDS addresses for importing to shipit
'    Authors:        Justin Morse, Ryan Diaz, Spee-Dee Delivery, Inc.
'    Changelog:      Date       Rev     Description of Changes
'                 ----------   -------  --------------------------------------------------------------------------------------------
'                 08/22/2025    0.1     Initial release
'				  09/24/2025    0.2		Fixed issue with reading phone number lines that only contain whitespace
'				  09/24/2025    0.3		Added check to make sure emails are in a valid format, fixed issue with writing first address
'                 ----------   -------  --------------------------------------------------------------------------------------------
'    Intro: This script will find the Addresses.mdb file used by SpeeDee Ship and export all the addresses. Then it will format them
'			so they can be imported into shipit.
'
'    Usage: Run SDS-Addresses-Export.bat file that is found in the same folder as this one.
'           
'=================================================================================================================================

' --- Force 32-bit execution ---
' In case the user doesn't have Access installed, use native provider for accessing the database
If InStr(LCase(WScript.FullName), "system32") > 0 Then
    Dim objShell, strCmd
    Set objShell = CreateObject("WScript.Shell")
    strCmd = Replace(LCase(WScript.FullName), "system32", "syswow64") & " """ & WScript.ScriptFullName & """"
    objShell.Run strCmd
    WScript.Quit
End If

'=================================================================================================================================
'Decalre Variables
'=================================================================================================================================

Dim accessApp, addressesPath, exportPath
Dim isNetworkInstall, settingsFile, readSettings, settingsPath, currentLine, lineLength
Dim addressLines, lineQuery, line, newColumnOrder
Dim csvApp, inputFile, outputFile, shipitCSVPath, oldColumns, newColumns, newHeaders
Dim mdbConnection, sqlQuery, mdbQuery, csvFile, exportFile, fieldTemp

'=================================================================================================================================
'Static Variables
'=================================================================================================================================

settingsPath = "C:\SPEEDEE\Settings.ini"
addressesPath = "C:\SPEEDEE\Addresses.mdb"
exportPath = "C:\SPEEDEE\Addresses_Export.csv"
shipitCSVPath = "C:\SPEEDEE\shipit_Import.csv"
sqlQuery = "SELECT * FROM Customers"


' Open Settings.ini file
Set settingsFile = CreateObject("Scripting.FileSystemObject")
Set readSettings = settingsFile.OpenTextFile(SettingsPath)

' Look for Addresses= line based on the regex
Set objRegex = CreateObject("VBScript.RegExp")
objRegEx.Pattern = "^Addresses=.+"

' Loop through each line looking for the Addresses= line
Do
	' Read current line
	currentLine = readSettings.ReadLine
	if objRegex.Test(currentLine) Then

		' If the Addresses= line is found, most likely a network install -> change where we're looking for the database
		lineLength = Len(currentLine)
		addressesPath = Mid(currentLine, 11, lineLength)
		Exit Do
	End If
Loop Until readSettings.AtEndOfStream

' Export addresses in database
' Columns are:
' 0				1			2			3			4			5		6		7	8		9		10		11			12
' CustomerNo	ShipName	Attention	Address1	Address2	City	State	Zip	Phone	E-mail	Record	LastUsed	Country

' Open mdb file
Set mdbConnection = CreateObject("ADODB.Connection")
mdbConnection.Open "Provider=Microsoft.Jet.OLEDB.4.0;Data Source=" & addressesPath & ";"

' Get data from file
Set mdbQuery = mdbConnection.Execute(sqlQuery)

' Create new file
Set csvFile = CreateObject("Scripting.FileSystemObject")
Set exportFile = csvFile.CreateTextFile(exportPath)

' Write to output file
Do Until mdbQuery.EOF
	For i = 0 To mdbQuery.Fields.Count - 1

		' Get each field, make sure it's a string and remove commas
		fieldTemp = mdbQuery.Fields(i) & ""
		fieldTemp = Replace(fieldTemp, ",", "")

		' Add each field to the line, with commas between them
		line = line & fieldTemp & ","

	Next

	' Write each line to the output file
	exportFile.WriteLine line
	mdbQuery.MoveNext
	line = ""
Loop

' Change column structure to:
' 0					1				2		3				4		5				6					7					8		9						10			11		12				13		14						
' Address Line 1	Address Line 2	Company	State/Province	City	Zip/Postal Code	Address Latitude	Address Longitude	Notes	Load/Unload Time (min)	Account Id	Name	Phone Number	Email	Language (2 letters code)
' Column 13 will need to be a new, empty column, otherwise they relate to the customer export columns
newColumnOrder = Array(3, 4, 13, 6, 5, 7, 13, 13, 2, 13, 13, 1, 8, 9, 13)
newHeaders = "Address Line 1,Address Line 2,Company,State/Province,City,Zip/Postal Code,Address Latitude,Address Longitude,Notes,Load/Unload Time (min),Account Id,Name,Phone Number,Email,Language (2 letters code)"

' Read exported file
Set csvApp = CreateObject("Scripting.FileSystemObject")
Set inputFile = csvApp.OpenTextFile(exportPath)
Set outputFile = csvApp.CreateTextFile(shipitCSVPath)

' First write the new headers to the output file
outputFile.WriteLine newHeaders

' Set new regex pattern to look for correct emails
objRegEx.Pattern = "^[A-Z0-9._%+-]+@[A-Z0-9.-]+\.[A-Z]{2,}$"
objRegex.IgnoreCase = True

Do Until inputFile.AtEndOfStream
	
	' Read current line
	currentLine = inputFile.ReadLine

	' Add a comma to the end of the line to create a new blank column
	currentLine = currentLine & ","

	' Split line into columns, comma separated
	oldColumns = Split(currentLine, ",")

	' Reorder columns to new format
	ReDim newColumns(UBound(newColumnOrder))
	For i = 0 To UBound(newColumnOrder)
		newColumns(i) = oldColumns(newColumnOrder(i))
	Next

	' Set language
	newColumns(14) = "en"

	' Add a phone number if there is none, check for entries containing only whitespace
	If Trim(newColumns(12)) = "" Then
		newColumns(12) = "000-000-0000"
	End If

	' Make sure emails are in a valid format: example@something.whatever
	' If email is invalid, remove it
	' If no email, skip
	if objRegex.Test(Trim(newColumns(13))) = False And Not Trim(newColumns(13)) = "" Then
		newColumns(13) = ""
	End If

	' Write new line to output file, separate columns by commas
	outputFile.WriteLine Join(newColumns, ",")
Loop

' Close files after use
inputFile.Close
outputFile.Close
exportFile.Close
mdbConnection.Close

' Delete the intermediate export file before releasing the FileSystemObject
If csvFile.FileExists(exportPath) Then
    csvFile.DeleteFile exportPath, True
Else
    MsgBox "No export file found to delete at: " & exportPath, vbExclamation, "Cleanup Skipped"
End If

csvApp.Close
Set exportFile = Nothing
Set mdbConnection = Nothing
Set csvApp = Nothing
Set csvFile = Nothing
Set inputFile = Nothing
Set outputFile = Nothing